//=================================================================== 
// COPYRIGHT T-Systems GmbH 2006
//===================================================================
//
// CMIToolboxApi.h
// Header definition of CMIToolboxApi
//
//===================================================================
//
// Usage notes:
//
// You have to use this API within the CMI CATIA session.
// Do not use this API within an external application.
//
//===================================================================

#ifndef CMIToolboxApi_H
#define CMIToolboxApi_H



/********************************************************************************
 * CMIToolboxApi
 *
 * You have to use this API within the interactive CATIA session. 
 * Do not use this API within an external application. 
 * The CMI Toolbox API uses the address space of the interactive CATIA session.
 *
 *
 * General usage:
 *
 * To use the global function CMIGetPartAttributeFromTC or 
 * CMIGetPartAttributeFromTC in your application you have to set CMICommands as 
 * prerequisite in your Imakefile. This causes that the CATIA V5 part of CMI has 
 * to be installed in the runtime environment of your application.
 *
 * To create an application that can either run with the CATIA V5 part of CMI 
 * installed in the runtime environment or not you have to use the APIs like 
 * follows:
 * 
 *  #include "CATGetEntryPoint.h"
 *  typedef HRESULT (*GetPartAttributeFromTC_FunkType)(CATIProduct_var, 
 *          const CATString&, CATString&,CATString&);
 *
 *  ...
 *
 *	LibraryHandler hGetPartAttributeFromTC = CATGetFunctionAddress (
 *			"CMICommands",
 *			"CMIGetPartAttributeFromTC");
 *	if (hGetPartAttributeFromTC.EntryPoint == NULL)
 *	{
 *     // CMI not installed  
 *     // You cannot call the CMI APIs in the current session
 *	   ...
 *	}
 *  else
 *  {
 *     // CMI installed
 *  ...
 *
 *     GetPartAttributeFromTC_FunkType
 *     pFunc = (GetPartAttributeFromTC)hGetPartAttributeFromTC.EntryPoint;
 *
 *	   hr = pFunc(spProduct,sAttribute,sValue,sReturnCode);
 *
 *  }
 *
 ********************************************************************************
 */

#ifdef  _WINDOWS_SOURCE
#define ExportedByCMIToolboxApi     __declspec(dllexport)
#else
#define ExportedByCMIToolboxApi
#endif

#include "IUnknown.h"
#include "CATBoolean.h"
class CATIProduct_var;
class CATString;
class CATDocument;

#ifdef __cplusplus
extern "C" 
{
#endif


/********************************************************************************
 *
 *   CMIGetPartAttributeFromTC
 *
 *   Get Attribute from Teamcenter. 
 *   This Funktion calls x3GetAttValueForObject in Teamcenter. 
 *
 *   In:      spProduct          CATPart / CATProduct to examine in Teamcenter  
 *            sAttribute         sAttributeName in x3GetAttValueForObject
 *
 *   Out:     sValue             sAttributeValue in x3GetAttValueForObject
 *            sReturnCode        sReturnCode in x3GetAttValueForObject
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... Product not loaded by CMI
 *                               E_FAIL   ... spProduct = NULL_var,
 *                                            Connetion to Teamcenter failed,
 *                                            internal Error
 *
 ********************************************************************************
 */


    ExportedByCMIToolboxApi 
	HRESULT CMIGetPartAttributeFromTC(const CATIProduct_var spProduct,
                                      const CATString &sAttribute, 
                                      CATString &sValue,
                                      CATString &sReturnCode); 


/********************************************************************************
 *
 *   CMIGetDrawingAttributeFromTC
 *
 *   Get Attribute from Teamcenter. 
 *   This Funktion calls x3GetAttValueForObject in Teamcenter. 
 *
 *   In:      pDocument          CATDrawing to examine in Teamcenter 
 *            sAttribute         sAttributeName in x3GetAttValueForObject
 *
 *   Out:     sValue             sAttributeValue in x3GetAttValueForObject
 *            sReturnCode        sReturnCode in x3GetAttValueForObject
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... CATDrawing not loaded by CMI,
 *                                            Wrong DocumentType detected
 *                               E_FAIL   ... pDocument = NULL,
 *                                            Connetion to Teamcenter failed,
 *                                            internal error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMIGetDrawingAttributeFromTC(const CATDocument *pDocument,
	                                     const CATString &sAttribute, 
	                                     CATString &sValue,
	                                     CATString &sReturnCode);


/********************************************************************************
 *
 *   CMISupplyPartAttributeToTC
 *
 *   Supply Attribute to Teamcenter. 
 *   This Funktion calls x3PutAttValueForObject in Teamcenter. 
 *
 *   In:      spProduct          CATPart / CATProduct to examine in Teamcenter  
 *            sAttribute         sAttributeName in x3PutAttValueForObject
 *            sValue             sAttributeValue in x3PutAttValueForObject
 *        
 *   Out:     bSuccess           Success status of Teamcenter 
 *                                            TRUE    ...  OK
 *                                            FALSE   ...  FAILED
 *            sReturnCode        sReturnCode in x3PutAttValueForObject
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... Product not loaded by CMI
 *                               E_FAIL   ... spProduct = NULL_var,
 *                                            Connetion to Teamcenter failed,
 *                                            internal Error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMISupplyPartAttributeToTC(const CATIProduct_var spProduct,
                                       const CATString &sAttribute, 
                                       const CATString &sValue,
                                       CATBoolean &bSuccess,
                                       CATString &sReturnCode);


/********************************************************************************
 *
 *   CMISupplyDrawingAttributeToTC
 *
 *   Get Attribute from Teamcenter. 
 *   This Funktion calls x3PutAttValueForObject in Teamcenter. 
 *
 *   In:      pDocument          CATDrawing to examine in Teamcenter  
 *            sAttribute         sAttributeName in x3PutAttValueForObject
 *            sValue             sAttributeValue in x3PutAttValueForObject
 *
 *   Out:     bSuccess           Success status of Teamcenter 
 *                                            TRUE    ...  OK
 *                                            FALSE   ...  FAILED
 *            sReturnCode        sReturnCode in x3PutAttValueForObject
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... CATDrawing not loaded by CMI,
 *                                            Wrong DocumentType detected
 *                               E_FAIL   ... pDocument = NULL,
 *                                            Connetion to Teamcenter failed,
 *                                            internal error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMISupplyDrawingAttributeToTC(const CATDocument *pDocument,
                                          const CATString &sAttribute, 
                                          const CATString &sValue,
                                          CATBoolean &bSuccess,
                                          CATString &sReturnCode);


/********************************************************************************
 *
 *   CMISupplyAdditionalFileForPrdToTC
 *
 *   Supply additional file for CATProduct / CATPart to Teamcenter
 *   This Funktion calls x3GetSupplyFilePath and x3StoreSupplyFileCus in Teamcenter. 
 *
 *   In:      spProduct          CATPart / CATProduct to examine in Teamcenter 
 *                               (can be NULL_var)
 *            sFullPath          full path of the additional file
 *            sInfoForTC         optional info for TC
 *
 *   Out:     bSuccess           Success status of Teamcenter 
 *                                            TRUE    ...  OK
 *                                            FALSE   ...  FAILED
 *                               sReturnCode  sReturnCode from Teamcenter
 *                                            This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               E_FAIL   ... Connetion to Teamcenter failed,
 *                                            internal error
 *
 ********************************************************************************
 */
	
	ExportedByCMIToolboxApi 
	HRESULT CMISupplyAdditionalFileForPrdToTC(const CATIProduct_var spProduct,
	                                          const CATString &sFullPath,
	                                          const CATString &sInfoForTC,
											  CATBoolean &bSuccess,
	                                          CATString &sReturnCode);


/********************************************************************************
 *
 *   CMISupplyAdditionalFileForDrwToTC
 *
 *   Supply additional file for CATDrawing to Teamcenter
 *   This Funktion calls x3GetSupplyFilePath and x3StoreSupplyFileCus in Teamcenter. 
 *
 *   In:      pDocument          CATDrawing to examine in Teamcenter 
 *                               (can be NULL)
 *            sFullPath          full path of the additional file
 *            sInfoForTC         optional info for TC
 *
 *   Out:     bSuccess           Success status of Teamcenter 
 *                                            TRUE    ... OK
 *                                            FALSE   ... FAILED
 *                               sReturnCode  sReturnCode from Teamcenter
 *                                            This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... Wrong DocumentType detected
 *                               E_FAIL   ... Connetion to Teamcenter failed,
 *                                            internal error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMISupplyAdditionalFileForDrwToTC(const CATDocument *pDocument,
	                                          const CATString &sFullPath,
	                                          const CATString &sInfoForTC,
											  CATBoolean &bSuccess,
	                                          CATString &sReturnCode);


/********************************************************************************
 *
 *   CMIInCMIArchive
 *
 *   Check if a CATIProduct is part of a CMIArchive
 *
 *   In:      spProduct          CATPart / CATProduct to examine
 *
 *   Out:     bInCMIArchive      status
 *                               TRUE    ...  CATIProduct is part of CMIArchive
 *                               FALSE   ...  CATIProduct is not part of CMIArchive
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... Product not loaded by CMI
 *                               E_FAIL   ... internal error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMIInCMIArchive(const CATIProduct_var spProduct, CATBoolean &bInCMIArchive);


/********************************************************************************
 *
 *   CMISetCMIMetaData
 *
 *   Load CATPart in design mode and set CMI Meta Data on CATPart if required.
 *
 *   By default CMI sets Meta Data like PartNumber, Nomenclature, Revision, 
 *   User Defined Properties, ... during CMI Read. This is not possible if 
 *   CATParts are loaded in visualization mode. For these CATParts CMI postpones 
 *   this action and trys again if any other CMI action is performed.
 *
 *   This API forces to set the CMI Meta Data on the investigated CATPart.
 *   This API should be used if your application depends on CMI Meta Data.
 *
 *   In:      spProduct          CATPart / CATProduct to investigated
 *
 *
 *   Return:                     S_OK     ... OK
 *                               E_FAIL   ... internal error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMISetCMIMetaData(CATIProduct_var &spProduct);




/********************************************************************************
 *
 *   CMIGetInstanceAttributeFromTC
 *
 *   Get Attribute from Teamcenter. 
 *   This Funktion calls x3GetAttValueForObject in Teamcenter. 
 *
 *   In:      spProduct          Instance CATPart / CATProduct to examine in Teamcenter  
 *            sAttribute         sAttributeName in x3GetAttValueForInstance
 *
 *   Out:     sValue             sAttributeValue in x3GetAttValueForObject
 *            sReturnCode        sReturnCode in x3GetAttValueForObject
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... Product instance not loaded by CMI
 *                               S_FALSE  ... Product instance not identifyable in TC
 *                               E_FAIL   ... spProduct == NULL_var,
 *                                            spProduct == ReferenceProduct
 *                                            Connetion to Teamcenter failed,
 *                                            internal Error
 *
 ********************************************************************************
 */


	ExportedByCMIToolboxApi 
	HRESULT CMIGetInstanceAttributeFromTC(const CATIProduct_var spProduct,
                                      const CATString &sAttribute, 
                                      CATString &sValue,
                                      CATString &sReturnCode); 



/********************************************************************************
 *
 *   CMISupplyInstanceAttributeToTC
 *
 *   Supply Attribute to Teamcenter. 
 *   This Funktion calls x3SupplyAttValueForObject in Teamcenter. 
 *
 *   In:      spProduct          Instance CATPart / CATProduct to examine in Teamcenter  
 *            sAttribute         sAttributeName in x3SupplyAttValueForInstance
 *            sValue             sAttributeValue in x3SupplyAttValueForInstance
 *        
 *   Out:     bSuccess           Success status of Teamcenter 
 *                                            TRUE    ...  OK
 *                                            FALSE   ...  FAILED
 *            sReturnCode        sReturnCode in x3SupplyAttValueForInstance
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               S_FALSE  ... Product instance not loaded by CMI
 *                               S_FALSE  ... Product instance not identifyable in TC
 *                               E_FAIL   ... spProduct == NULL_var,
 *                                            spProduct == ReferenceProduct
 *                                            Connetion to Teamcenter failed,
 *                                            internal Error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMISupplyInstanceAttributeToTC(const CATIProduct_var spProduct,
                                       const CATString &sAttribute, 
                                       const CATString &sValue,
                                       CATBoolean &bSuccess,
                                       CATString &sReturnCode);


/********************************************************************************
 *
 *   CMIOpenPartFromTeamcenter
 *
 *   Query for Part in Teamcenter and open the expanded structure in CATIA. 
 *   This Funktion calls x3GetPartForCatiaApi in Teamcenter. 
 *
 *   In:      sPartnumber        Partnumber of expected Part (required) 
 *            sClass             Classname of expected Part (can be an empty String)
 *            sRevision          Revision of expected Part (can be an empty String)
 *            sSequence          Sequence of expected Part (can be an empty String)
 *            spParentProduct    Parent CATProduct to insert downloaded structure.
 *                               The Parent Product has to be an Instance Product
 *                               (if NULL_var, open downloaded structure in own window)
 *        
 *   Out:     bSuccess           Success status of Teamcenter 
 *                                            TRUE    ...  OK
 *                                            FALSE   ...  FAILED
 *            sReturnCode        sReturnCode in x3GetPartForCatiaApi
 *                               This holds the failure Message from Teamcenter
 *
 *   Return:                     S_OK     ... OK
 *                               E_FAIL   ... Connetion to Teamcenter failed,
 *                                            internal Error
 *
 ********************************************************************************
 */

	ExportedByCMIToolboxApi 
	HRESULT CMIOpenPartFromTeamcenter( const CATString &sPartnumber,
	                                   const CATString &sClass,
	                                   const CATString &sRevision,
	                                   const CATString &sSequence,
	                                   CATIProduct_var &spParentProduct,
	                                   CATBoolean &bSuccess,
   	                                   CATString &sReturnCode);



#ifdef __cplusplus
}
#endif

#endif   // CMIToolboxApi_H
